/* Copyright (c) 2010 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is property of Nordic Semiconductor ASA.
 * Terms and conditions of usage are described in detail in NORDIC
 * SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT. 
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 */ 

/** @file
 * @brief Implementation of lib_ir_transmit
 */

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include "nrf24le1.h"
#include "lib_ir_transmit.h"
#include "nordic_common.h"

// Internal variables
static uint16_t timer_counter;
static uint16_t sequence_counter;
static uint8_t bit_counter;
static uint8_t pulse_counter;
static uint8_t repeat_counter;

static uint16_t gap_target;
static uint16_t timer_target;
static uint8_t timer_period;

static uint32_t data_sequence;

static uint8_t transmit_flags;
static bool transmit;
static bool repeat_ready;
static bool repeat;
static bool first_run;
         
static uint8_t pwm_freq;
static uint8_t pwm_duty_cycle;

static lib_ir_protocol_struct_t current_protocol;

// Internal functions

/** Function to set the current protocol
 *  @param protocol The name of the protocol.
 */          
static void set_protocol(lib_ir_protocol_name_t protocol);

/** Function to set the duty cycle of the PWM
 *  This function calculates the correct value for the PWM register
 *  @param duty_cycle The duty cycle in percent.
 */  
static void set_pwm_duty_cycle(uint8_t duty_cycle);

/** Function to set the frequency of the PWM
 *  This function calculates the correct value and configures the PWM for the correct frequency
 *  @param freq The frequency in Hz.
 */  
static void set_pwm_frequency(uint16_t freq);

/** Function to initiate a transmission
 *  This function clears some variables and sets the transmit flags
 */         
static void initiate_transmission(void);

/** Function to control the PWM on or off
 *  @param start States whether to start or stop the PWM
 */
static void pwm_on(bool start);   
           
void lib_ir_transmit_init(uint8_t tp)
{ 
  // Save the timer period
  timer_period = tp;
  
  // Initialize a ~38kHz PWM
  PWMCON = 0x37;
  LIB_IR_PWM_OUTPUT(0x00);// = 0x00;     // Turn off PWM outputs

  // Clear counters
  transmit = false;
  timer_counter = 0;
  timer_target = 0;
  sequence_counter = 0;
  data_sequence = 0;
  bit_counter = 0;
  pulse_counter = 0;

  // Clear the protocol
  set_protocol(PROTOCOL_NONE);
}

static void set_protocol(lib_ir_protocol_name_t protocol)
{ 
  // Set the correct protocol 
  switch (protocol)
  {
    case PROTOCOL_NONE:
    {
      memset(&current_protocol, 0x00, sizeof(lib_ir_protocol_struct_t));
      current_protocol.protocol = PROTOCOL_NONE;
      break;
    } 
    case PROTOCOL_NEC:
    {
      current_protocol = lib_ir_protocol_nec;
      break;
    }     
    case PROTOCOL_RC5:
    {
      current_protocol = lib_ir_protocol_rc5;
      break;
    }
    case PROTOCOL_SIRC_12BIT:
    {
      current_protocol = lib_ir_protocol_sirc_12bit;
      break;
    }
    case PROTOCOL_SIRC_15BIT:
    {
      current_protocol = lib_ir_protocol_sirc_15bit;
      break;
    } 
    case PROTOCOL_SIRC_20BIT:
    {
      current_protocol = lib_ir_protocol_sirc_20bit;
      break;
    } 
    default:
    {
      memset(&current_protocol, 0x00, sizeof(lib_ir_protocol_struct_t));
      break;
    }
  }
  // Set the correct duty cycle and frequency
  set_pwm_duty_cycle(current_protocol.duty_cycle);
  set_pwm_frequency(current_protocol.frequency);

  // Calculate the timer target for the gap
  gap_target = (uint16_t)(current_protocol.gap / timer_period);
  sequence_counter = 0;
}

void set_pwm_duty_cycle(uint8_t duty_cycle)
{
  // Calculate the configuration parameter for the PWM
  uint16_t tmp = (duty_cycle * 31);

  pwm_duty_cycle = (tmp / 100);
  if (tmp % 100 > 0)
  {
    if (100 / (tmp % 100) < 2)
        pwm_duty_cycle += 1;
    else if (100 / (tmp % 100) == 2 && 100 % (tmp % 100) == 0)
        pwm_duty_cycle += 1;
  }
}

void set_pwm_frequency(uint16_t freq)
{
  // Calculate the configuration parameter for the PWM and set the correct frequency
  uint16_t tmp = (freq / 1000) * 31;

  pwm_freq = (16000 / tmp);
  if (tmp / (16000 % tmp) < 2)
    pwm_freq += 1;

  pwm_freq -= 1;

  PWMCON = (PWMCON & 0xC3) | ((pwm_freq & 0x0F) << 2);
  LIB_IR_PWM_OUTPUT(0x00);       // Turn off PWM outputs
}

bool lib_ir_transmit_nec_command(uint8_t addr, uint8_t inv_addr, uint8_t command, uint8_t inv_command, bool rep)
{
  // Return if an transmission or reception is already in progress
  if (transmit)
    return false;

  if (rep)
  {
    // Return if the command is a repeated command and the gap time has not elapsed
    if (!repeat_ready)
      return false;

    // Set the correct protocol if needed
    if (current_protocol.protocol != PROTOCOL_NEC)
      set_protocol(PROTOCOL_NEC);

    // State which part of the packet we are transmitting
    transmit_flags = REPEAT | TRAIL;
  }
  else
  {
    // Build the data sequence
    data_sequence = 0;                                    // Clear the data sequence
    data_sequence |= ((uint32_t)(addr & 0xFF)) << 24;     // Add the address
    data_sequence |= ((uint32_t)(inv_addr & 0xFF)) << 16; // Add the inverted address
    data_sequence |= ((uint32_t)(command & 0xFF)) << 8;   // Add the command
    data_sequence |= ((uint32_t)(inv_command & 0xFF));    // Add the inverted command
  
    // Set the correct protocol if needed
    if (current_protocol.protocol != PROTOCOL_NEC)
      set_protocol(PROTOCOL_NEC);
    else if (!repeat_ready)
      return false;

    // State which part of the packet we are transmitting
    transmit_flags = current_protocol.flags;
  }

  // Start the transmission
  initiate_transmission();
  return true;
}

bool lib_ir_transmit_sirc_command(uint8_t addr, uint8_t command, uint8_t extended, lib_ir_protocol_name_t version, bool rep)
{
  // Return if an transmission or reception is already in progress
  if (transmit)
    return false;

  // Build the data sequence      
  data_sequence = 0;                                    
  if (version == PROTOCOL_SIRC_12BIT)
  { 
    data_sequence |= ((uint32_t)command & 0x7F) << 5;   // Add the address
    data_sequence |= (uint32_t)addr & 0x1F;             // Add the command
  }
  else if (version == PROTOCOL_SIRC_15BIT)
  {
    data_sequence |= ((uint32_t)command & 0x7F) << 8;   // Add the address
    data_sequence |= (uint32_t)addr & 0xFF;             // Add the command
  }
  else if (version == PROTOCOL_SIRC_20BIT)
  {
    data_sequence |= ((uint32_t)command & 0x7F) << 13;  // Add the address
    data_sequence |= ((uint32_t)addr & 0x1F) << 8;      // Add the command
    data_sequence |= (uint32_t)extended & 0xFF;         // Add the extended word
  }    

  // Set the correct protocol if needed
  if (current_protocol.protocol != version)
    set_protocol(version);
  else if (!repeat_ready)
    return false;

  // State which part of the packet we are transmitting
  transmit_flags = current_protocol.flags;

  // Return if the command is a repeated command and the gap time has not elapsed
  if (rep && !repeat_ready)
    return false;

  // Start the transmission
  initiate_transmission();
  return true;
}

bool lib_ir_transmit_rc5_command(uint8_t addr, uint8_t command, bool rep)
{
  static bit rc5_toggle_bit;

  // Return if an transmission or reception is already in progress
  if (transmit)
    return false;

  // If this is not a repeated command, invert the toggle bit
  if (!rep)
    rc5_toggle_bit = !rc5_toggle_bit;
  
  // Set the correct protocol if needed
  if (current_protocol.protocol != PROTOCOL_RC5)
    set_protocol(PROTOCOL_RC5);
  else if (!repeat_ready)
    return false;

  // Build the data sequence
  data_sequence = 0x00001000;                           // The start bit
  data_sequence |= ((uint32_t)rc5_toggle_bit & 0x01) << ((current_protocol.bits - 1) - current_protocol.toggle_bit); // Add the toggle bit
  data_sequence |= ((uint32_t)addr & 0x1F) << 6;        // Add the address
  data_sequence |= (uint32_t)command & 0x3F;            // Add the command
  
  // State which part of the packet we are transmitting
  transmit_flags = current_protocol.flags;

  // Return if the command is a repeated command and the gap time has not elapsed
  if (rep && !repeat_ready)
    return false;

  // Start the transmission
  initiate_transmission();
  return true;
}

void initiate_transmission(void)
{
  // Set the number of bits we want to transmit and the number of repetitions
  bit_counter = current_protocol.bits - 1;
  repeat_counter = current_protocol.min_repeat;

  first_run = true;
  repeat = false;
  repeat_ready = false;
  pulse_counter = 0;
  sequence_counter = 0;

  transmit = true;
}

void pwm_on(bool start)
{
  if (start)
  {
    // Turn on the PWM, with the correct duty cycle
    LIB_IR_PWM_OUTPUT(pwm_duty_cycle);
  }
  else
  {
    // Turn off the PWM output
    LIB_IR_PWM_OUTPUT(0x00);
  }
}

void lib_ir_transmit_timer_irq_function(void)
{
  // Increment counters
  sequence_counter++;
  timer_counter++;

  // If the gap time has elapsed: Set flag to indicate that a new sequence can be sent
  if (sequence_counter == gap_target)
  {
    repeat_ready = true;
    sequence_counter = 0;
  }

  // Return if the gap time has not elapsed and repeated sequences are waiting
  if(repeat && !repeat_ready)
    return;

  if (repeat)
  {
    repeat = false;
    repeat_ready = false;
  }

  // If not in transmit mode: Return from the function
  if(!transmit)
    return;

  // Check if this is the first run or the timer has reached the target
  if (first_run || timer_counter == timer_target)
  {   
    if (first_run)
    {
      sequence_counter = 0;  
      repeat_ready = false;
    }
    
    first_run = false;        // Clear first run flag 
    timer_counter = 0;        // Reset counter

    // Check if the header flag is set
    if (transmit_flags & HEADER)
    {
      // Check if the pulse counter is less than 2
      if (pulse_counter < 2)
      {
        // Turn on PWM output if the number is positive, turn off PWM output if the number is negative
        // Set the correct timer target and increment pulse counter
        pwm_on(current_protocol.header[pulse_counter] > 0);
        timer_target = (uint16_t)(abs(current_protocol.header[pulse_counter]) / timer_period);
        pulse_counter++;
      }
      // If the pulse counter is 2; clear the counter and the header flag
      if (pulse_counter == 2)
      {
        pulse_counter = 0;
        transmit_flags &= ~HEADER;
      }
    }

    // Check if the trail flag is set
    else if (transmit_flags & LEAD)
    {
      // Turn on PWM output if the number is positive, turn off PWM output if the number is negative
      // Set the correct timer target and clear the counter and the trail flag
      pwm_on(current_protocol.lead > 0);
      timer_target = (uint16_t)(abs(current_protocol.lead) / timer_period);
      transmit_flags &= ~LEAD;
    }

    // Check if the data flag is set
    else if (transmit_flags & DATA)
    {
      // Check if the pulse counter is less than 2
      if (pulse_counter < 2)
      {
        // Send the correct bit from the data sequence
        if ((data_sequence >> bit_counter) & 0x01 == 1)
        {
          pwm_on(current_protocol.one[pulse_counter] > 0);
          timer_target = (uint16_t)(abs(current_protocol.one[pulse_counter]) / timer_period);
        }
        else
        {
          pwm_on(current_protocol.zero[pulse_counter] > 0);
          timer_target = (uint16_t)(abs(current_protocol.zero[pulse_counter]) / timer_period);
        }        

        // Increment pulse counter 
        pulse_counter++;
      }

      // If the pulse counter is 2; clear the counter
      if (pulse_counter == 2)
      {
        pulse_counter = 0;

        // If the last bit is transmitted; clear the data flag,
        // else decrement the bit counter
        if (bit_counter == 0)
          transmit_flags &= ~DATA;
        else
          bit_counter--;
      }
    }

    // Check if the repeat flag is set
    else if (transmit_flags & REPEAT)
    {
      // Check if the pulse counter is less than 2
      if (pulse_counter < 2)
      {
        // Turn on PWM output if the number is positive, turn off PWM output if the number is negative
        // Set the correct timer target and increment pulse counter
        pwm_on(current_protocol.repeat[pulse_counter] > 0);
        timer_target = (uint16_t)(abs(current_protocol.repeat[pulse_counter]) / timer_period);
        pulse_counter++;
      }

      // If the pulse counter is 2; clear the counter and the repeat flag
      if (pulse_counter == 2)
      {
        pulse_counter = 0;
        transmit_flags &= ~REPEAT;
      }
    }

    // Check if the trail flag is set
    else if (transmit_flags & TRAIL)
    {
      // Turn on PWM output if the number is positive, turn off PWM output if the number is negative
      // Set the correct timer target and clear the counter and the trail flag
      pwm_on(current_protocol.trail > 0);
      timer_target = (uint16_t)(abs(current_protocol.trail) / timer_period);
      transmit_flags &= ~TRAIL;
    }

    else
    {
      // Turn off PWM outputs 
      pwm_on(false);

      // If there is a minimum number of repeated sequences, reset the flags and the bit counter                             
      if (repeat_counter > 0)
      {
        repeat_counter--;
        transmit_flags = current_protocol.flags;
        bit_counter = current_protocol.bits - 1;
        repeat = true;
        first_run = true;
        pulse_counter = 0;
      }

      // Reset the flags and give a call-back when the transmission is done
      else
      {
        repeat = false;
        lib_ir_transmit_data_sent_callback();
        transmit = false;
      }
    }
  }
}
